/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.jts.index.hprtree;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.index.ArrayListVisitor;
import org.locationtech.jts.index.ItemVisitor;
import org.locationtech.jts.index.SpatialIndex;
import org.locationtech.jts.index.hprtree.HilbertEncoder;
import org.locationtech.jts.index.hprtree.Item;

public class HPRtree
implements SpatialIndex {
    private static final int ENV_SIZE = 4;
    private static final int HILBERT_LEVEL = 12;
    private static int DEFAULT_NODE_CAPACITY = 16;
    private List<Item> items = new ArrayList<Item>();
    private int nodeCapacity = DEFAULT_NODE_CAPACITY;
    private Envelope totalExtent = new Envelope();
    private int[] layerStartIndex;
    private double[] nodeBounds;
    private boolean isBuilt = false;

    public HPRtree() {
        this(DEFAULT_NODE_CAPACITY);
    }

    public HPRtree(int nodeCapacity) {
        this.nodeCapacity = nodeCapacity;
    }

    public int size() {
        return this.items.size();
    }

    @Override
    public void insert(Envelope itemEnv, Object item) {
        if (this.isBuilt) {
            throw new IllegalStateException("Cannot insert items after tree is built.");
        }
        this.items.add(new Item(itemEnv, item));
        this.totalExtent.expandToInclude(itemEnv);
    }

    @Override
    public List query(Envelope searchEnv) {
        this.build();
        if (!this.totalExtent.intersects(searchEnv)) {
            return new ArrayList();
        }
        ArrayListVisitor visitor = new ArrayListVisitor();
        this.query(searchEnv, visitor);
        return visitor.getItems();
    }

    @Override
    public void query(Envelope searchEnv, ItemVisitor visitor) {
        this.build();
        if (!this.totalExtent.intersects(searchEnv)) {
            return;
        }
        if (this.layerStartIndex == null) {
            this.queryItems(0, searchEnv, visitor);
        } else {
            this.queryTopLayer(searchEnv, visitor);
        }
    }

    private void queryTopLayer(Envelope searchEnv, ItemVisitor visitor) {
        int layerIndex = this.layerStartIndex.length - 2;
        int layerSize = this.layerSize(layerIndex);
        int i = 0;
        while (i < layerSize) {
            this.queryNode(layerIndex, i, searchEnv, visitor);
            i += 4;
        }
    }

    private void queryNode(int layerIndex, int nodeOffset, Envelope searchEnv, ItemVisitor visitor) {
        int layerStart = this.layerStartIndex[layerIndex];
        int nodeIndex = layerStart + nodeOffset;
        if (!this.intersects(nodeIndex, searchEnv)) {
            return;
        }
        if (layerIndex == 0) {
            int childNodesOffset = nodeOffset / 4 * this.nodeCapacity;
            this.queryItems(childNodesOffset, searchEnv, visitor);
        } else {
            int childNodesOffset = nodeOffset * this.nodeCapacity;
            this.queryNodeChildren(layerIndex - 1, childNodesOffset, searchEnv, visitor);
        }
    }

    private boolean intersects(int nodeIndex, Envelope env) {
        boolean isBeyond = env.getMaxX() < this.nodeBounds[nodeIndex] || env.getMaxY() < this.nodeBounds[nodeIndex + 1] || env.getMinX() > this.nodeBounds[nodeIndex + 2] || env.getMinY() > this.nodeBounds[nodeIndex + 3];
        return !isBeyond;
    }

    private void queryNodeChildren(int layerIndex, int blockOffset, Envelope searchEnv, ItemVisitor visitor) {
        int layerStart = this.layerStartIndex[layerIndex];
        int layerEnd = this.layerStartIndex[layerIndex + 1];
        int i = 0;
        while (i < this.nodeCapacity) {
            int nodeOffset = blockOffset + 4 * i;
            if (layerStart + nodeOffset >= layerEnd) break;
            this.queryNode(layerIndex, nodeOffset, searchEnv, visitor);
            ++i;
        }
    }

    private void queryItems(int blockStart, Envelope searchEnv, ItemVisitor visitor) {
        int i = 0;
        while (i < this.nodeCapacity) {
            int itemIndex = blockStart + i;
            if (itemIndex >= this.items.size()) break;
            Item item = this.items.get(itemIndex);
            if (HPRtree.intersects(item.getEnvelope(), searchEnv)) {
                visitor.visitItem(item.getItem());
            }
            ++i;
        }
    }

    private static boolean intersects(Envelope env1, Envelope env2) {
        return !(env2.getMinX() > env1.getMaxX() || env2.getMaxX() < env1.getMinX() || env2.getMinY() > env1.getMaxY() || env2.getMaxY() < env1.getMinY());
    }

    private int layerSize(int layerIndex) {
        int layerStart = this.layerStartIndex[layerIndex];
        int layerEnd = this.layerStartIndex[layerIndex + 1];
        return layerEnd - layerStart;
    }

    @Override
    public boolean remove(Envelope itemEnv, Object item) {
        return false;
    }

    public synchronized void build() {
        if (this.isBuilt) {
            return;
        }
        this.isBuilt = true;
        if (this.items.size() <= this.nodeCapacity) {
            return;
        }
        this.sortItems();
        this.layerStartIndex = HPRtree.computeLayerIndices(this.items.size(), this.nodeCapacity);
        int nodeCount = this.layerStartIndex[this.layerStartIndex.length - 1] / 4;
        this.nodeBounds = HPRtree.createBoundsArray(nodeCount);
        this.computeLeafNodes(this.layerStartIndex[1]);
        int i = 1;
        while (i < this.layerStartIndex.length - 1) {
            this.computeLayerNodes(i);
            ++i;
        }
    }

    private static void dumpItems(List<Item> items) {
        GeometryFactory fact = new GeometryFactory();
        for (Item item : items) {
            Envelope env = item.getEnvelope();
            System.out.println(fact.toGeometry(env));
        }
    }

    private static double[] createBoundsArray(int size) {
        double[] a = new double[4 * size];
        int i = 0;
        while (i < size) {
            int index = 4 * i;
            a[index] = Double.MAX_VALUE;
            a[index + 1] = Double.MAX_VALUE;
            a[index + 2] = -1.7976931348623157E308;
            a[index + 3] = -1.7976931348623157E308;
            ++i;
        }
        return a;
    }

    private void computeLayerNodes(int layerIndex) {
        int layerStart = this.layerStartIndex[layerIndex];
        int childLayerStart = this.layerStartIndex[layerIndex - 1];
        int layerSize = this.layerSize(layerIndex);
        int childLayerEnd = layerStart;
        int i = 0;
        while (i < layerSize) {
            int childStart = childLayerStart + this.nodeCapacity * i;
            this.computeNodeBounds(layerStart + i, childStart, childLayerEnd);
            i += 4;
        }
    }

    private void computeNodeBounds(int nodeIndex, int blockStart, int nodeMaxIndex) {
        int i = 0;
        while (i <= this.nodeCapacity) {
            int index = blockStart + 4 * i;
            if (index >= nodeMaxIndex) break;
            this.updateNodeBounds(nodeIndex, this.nodeBounds[index], this.nodeBounds[index + 1], this.nodeBounds[index + 2], this.nodeBounds[index + 3]);
            ++i;
        }
    }

    private void computeLeafNodes(int layerSize) {
        int i = 0;
        while (i < layerSize) {
            this.computeLeafNodeBounds(i, this.nodeCapacity * i / 4);
            i += 4;
        }
    }

    private void computeLeafNodeBounds(int nodeIndex, int blockStart) {
        int i = 0;
        while (i <= this.nodeCapacity) {
            int itemIndex = blockStart + i;
            if (itemIndex >= this.items.size()) break;
            Envelope env = this.items.get(itemIndex).getEnvelope();
            this.updateNodeBounds(nodeIndex, env.getMinX(), env.getMinY(), env.getMaxX(), env.getMaxY());
            ++i;
        }
    }

    private void updateNodeBounds(int nodeIndex, double minX, double minY, double maxX, double maxY) {
        if (minX < this.nodeBounds[nodeIndex]) {
            this.nodeBounds[nodeIndex] = minX;
        }
        if (minY < this.nodeBounds[nodeIndex + 1]) {
            this.nodeBounds[nodeIndex + 1] = minY;
        }
        if (maxX > this.nodeBounds[nodeIndex + 2]) {
            this.nodeBounds[nodeIndex + 2] = maxX;
        }
        if (maxY > this.nodeBounds[nodeIndex + 3]) {
            this.nodeBounds[nodeIndex + 3] = maxY;
        }
    }

    private Envelope getNodeEnvelope(int i) {
        return new Envelope(this.nodeBounds[i], this.nodeBounds[i + 1], this.nodeBounds[i + 2], this.nodeBounds[i + 3]);
    }

    private static int[] computeLayerIndices(int itemSize, int nodeCapacity) {
        ArrayList<Integer> layerIndexList = new ArrayList<Integer>();
        int layerSize = itemSize;
        int index = 0;
        do {
            layerIndexList.add(index);
            layerSize = HPRtree.numNodesToCover(layerSize, nodeCapacity);
            index += 4 * layerSize;
        } while (layerSize > 1);
        return HPRtree.toIntArray(layerIndexList);
    }

    private static int numNodesToCover(int nChild, int nodeCapacity) {
        int mult = nChild / nodeCapacity;
        int total = mult * nodeCapacity;
        if (total == nChild) {
            return mult;
        }
        return mult + 1;
    }

    private static int[] toIntArray(List<Integer> list) {
        int[] array = new int[list.size()];
        int i = 0;
        while (i < array.length) {
            array[i] = list.get(i);
            ++i;
        }
        return array;
    }

    public Envelope[] getBounds() {
        int numNodes = this.nodeBounds.length / 4;
        Envelope[] bounds = new Envelope[numNodes];
        int i = numNodes - 1;
        while (i >= 0) {
            int boundIndex = 4 * i;
            bounds[i] = new Envelope(this.nodeBounds[boundIndex], this.nodeBounds[boundIndex + 2], this.nodeBounds[boundIndex + 1], this.nodeBounds[boundIndex + 3]);
            --i;
        }
        return bounds;
    }

    private void sortItems() {
        ItemComparator comp = new ItemComparator(new HilbertEncoder(12, this.totalExtent));
        Collections.sort(this.items, comp);
    }

    static class ItemComparator
    implements Comparator<Item> {
        private HilbertEncoder encoder;

        public ItemComparator(HilbertEncoder encoder) {
            this.encoder = encoder;
        }

        @Override
        public int compare(Item item1, Item item2) {
            int hcode1 = this.encoder.encode(item1.getEnvelope());
            int hcode2 = this.encoder.encode(item2.getEnvelope());
            return Integer.valueOf(hcode1).compareTo(hcode2);
        }
    }
}

